;;;
;;; diskcopy_image.lisp
;;;
;;; Read DiskCopy format disk images used to distribute Apple II 
;;; and old Macintosh disk images
;;;
;;; $Id$
;;;
;;; $Log$
;;;
;;;

;;; The DiskCopy disk image format was initially specified in the Apple
;;; II File Type Note E00005 (File Type $E0, auxiliary type $0005)
;;; On Macintosh, it was HFS file type dImg, creator dCpy
;;;
;;; DiskCopy was developed originally by Steve Christensen.
;;;

(defconstant +dc42-block-length+ 512 "Block length for Disk Copy 4.2 images.")
(defconstant +dc42-namelen-offset+ 0 
  "Length byte for Pascal-style namestring.")
(defconstant +dc42-name-offset+ 1)
(defconstant +dc42-name-maxlen+ 63)
(defconstant +dc42-datasize-offset+ 64)
(defconstant +dc42-tagsize-offset+ 68)
(defconstant +dc42-datacheck-offset+ 72)
(defconstant +dc42-tagcheck-offset+ 76)
(defconstant +dc42-diskform-offset+ 80)
(defconstant +dc42-formbyte-offset+ 81)
(defconstant +dc42-private-offset+ 82)
(defconstant +dc42-data-offset+ 84)

(defclass diskcopy-image (disk-image)
  ((stream :accessor stream :initarg :stream)))

(defun open-diskcopy-image (pathname)
  (let ((str (open pathname :direction :input
		   :element-type '(unsigned-byte 8))))
    (make-instance 'diskcopy-image :stream str)))

(defmethod diskcopy-name ((di diskcopy-image))
  (let ((pstring (make-array +dc42-name-maxlen+))
	(str (stream di)))
    (file-position str :start)
    (let* ((len (read-byte str))
	   (s (make-string len)))
      (read-sequence pstring str :start 0 :end (1- +dc42-name-maxlen+))
      (dotimes (i len s)
	(setf (aref s i) (code-char (aref pstring i)))))))

(defmethod image-byte ((di diskcopy-image) &optional byte-offset)
  "Returns the BYTE-OFFSETth byte of the raw image.
If byte-offset is NIL or not specifed, reads one byte at the 
current stream offset."
  (let ((str (stream di)))
    (if byte-offset (file-position str byte-offset))
    (read-byte str)))

(defmethod image-bytes ((di diskcopy-image) n &optional byte-offset)
  "Reads an integer beginning at the specified BYTE-OFFSET.
If none is specified, or byte-offset is NIL, begins at the current 
stream position. Least-significant byte first."
  (let ((str (stream di))
	(l 0))
    (if byte-offset
	(file-position str byte-offset))
    (dotimes (i n l)
      (setf (ldb (byte 8 (* 8 i)) l) (read-byte str)))))

(defmethod image-word ((di diskcopy-image) &optional byte-offset)
  (image-bytes di 2 byte-offset))

(defmethod image-long ((di diskcopy-image) &optional byte-offset)
  (image-bytes di 4 byte-offset))

(defmethod reverse-bytes ((di diskcopy-image) n &optional byte-offset)
  "Reads a 'reverse integer' beginning at the specified BYTE-OFFSET.
If none is specified, or byte-offset is NIL, begins at the current 
stream position.

A 'reverse integer' is a N-byte value, most-significant-byte first."

  (let ((str (stream di))
	(l 0))
    (if byte-offset
	(file-position str byte-offset))
    (dotimes (i n l)
      (setf (ldb (byte 8 (* 8 (- (1- n) i))) l) (read-byte str)))))

(defmethod reverse-long ((di diskcopy-image) &optional byte-offset)
  "A reverse long is four bytes, most significant byte first."
  (reverse-bytes di 4 byte-offset))

(defmethod reverse-word ((di diskcopy-image) &optional byte-offset)
  "A reverse word is two bytes, most significant byte first"
  (reverse-bytes di 2 byte-offset))

(defmethod datasize ((di diskcopy-image))
  (reverse-long di +dc42-datasize-offset+))

(defmethod tagsize ((di diskcopy-image))
  (reverse-long di +dc42-tagsize-offset+))

(defmethod diskformat ((di diskcopy-image))
  (let ((format-byte (image-byte di +dc42-diskform-offset+)))
    (case format-byte
      (0 :400k)
      (1 :800k)
      (2 :720k)
      (3 :1440k)
      (t format-byte))))

(defmethod formatbyte ((di diskcopy-image))
  (let ((fb (image-byte di +dc42-formbyte-offset+)))
    (case fb
      (#x12 :400k-format)
      (#x22 :macintosh)
      (#x24 :800k-apple-II)
      (t fb))))

(defmethod standard-private-word ((di diskcopy-image))
  (let ((pw (reverse-word di +dc42-private-offset+)))
    (values (= #x100 pw) pw)))


(defmethod recorded-data-checksum ((di diskcopy-image))
  "Returns the data checksum recorded in the file header."
  (reverse-long di +dc42-datacheck-offset+))

(defmethod recorded-tag-checksum ((di diskcopy-image))
  "Returns the tag checksum recorded in the file header."
  (reverse-long di +dc42-tagcheck-offset+))

(defmethod userdata-block ((di diskcopy-image) block-num)
  "Returns the data in the BLOCK-NUMth block (512 bytes long). 
The first block is BLOCK-NUM 0."
  (let ((str (stream di))
	(blk (make-array +dc42-block-length+)))
    (file-position str (+ +dc42-data-offset+
			  (* block-num +dc42-block-length+)))
    (read-sequence blk str :start 0 :end (1- +dc42-block-length+))
    blk))

(defmethod tagdata-block ((di diskcopy-image) block-num)
  "Returns the tag data for the BLOCK-NUMth block (12 bytes long).
The first block is BLOCK-NUM 0."
  (let ((str (stream di))
	(tags (make-array 12)))
    (file-position str (+ 84 (datasize di) (* 12 block-num)))
    (read-sequence tags str :start 0 :end 11)
    tags))

(defun inc-checksum (cs word)
  (let* ((sum (ldb (byte 32 0) (+ cs word))) ; modulo 32-bits
	 (bit (ldb (byte 1 0) sum))
	 (shifted-csum (ash sum -1))
	 (modified (dpb bit (byte 1 31) shifted-csum)))
    modified))

#|
(defun binf (n)
  (format nil "#b~32,'0B" n))
(defun hexf (n)
  (format nil "#x~8,'0X" n))
(defun test-checksum (cs word)
  (let ((inc (inc-checksum cs word)))
    (values inc (binf inc))))

|#

(defmethod inc-checksum-32-literal (cs word)
  "Word is 16-bits read MS byte first from file, cs is 32 bits.
   Try to do this as the 65c816 code does."
  (let ((low-cs (ldb (byte 16 0) cs))
	(high-cs (ldb (byte 16 16) cs)))
    (let* ((low-sum (+ low-cs word))
	   (low-result (ldb (byte 16 0) low-sum))
	   (low-carry (ldb (byte 1 16) low-sum))
	   (high-sum (ldb (byte 16 0) (+ low-carry high-cs))))

      ;; high carry gets lost?
      ;; depends on what INC <checksum+2 does... INC on 65c02 at least
      ;; does not affect carry, so I believe it does.

      ;; now rotate right the high-sum and low-sum
      
      (let ((lsb-high (ldb (byte 1 0) high-sum))
	    (rest-high (ldb (byte 15 1) high-sum))
	    (lsb-low (ldb (byte 1 0) low-result))
	    (rest-low (ldb (byte 15 1) low-result))
	    (new-sum 0))
	(setf new-sum
	      (dpb rest-low (byte 15 0) new-sum))
	(setf new-sum
	      (dpb lsb-high (byte 1 15) new-sum))
	(setf new-sum
	      (dpb rest-high (byte 15 16) new-sum))
	(setf new-sum
	      (dpb lsb-low (byte 1 31) new-sum))
	new-sum))))
		   
      
(defmethod calculate-checksum ((di diskcopy-image)
			       start-byte-offset 
			       num-words
			       &optional (initial-value 0))
  "The documented method is

Initialize (32-bit) checksum to zero [uses INITIAL-VALUE]
For each data reverse-word 
  add the data reverse-word to the checksum
  rotate the 32-bit checksum right one bit (bit 0 wraps to bit 31)."

  (let ((csum initial-value)
	(str (stream di)))
    (file-position str start-byte-offset)
    (dotimes (i num-words csum)
      (setf csum (inc-checksum csum (reverse-word di nil))))))

;;; CAN'T get this to agree with the IIgs system 6.0 disks...
;;; reason was that the image data had been modified by Kegs
;;; on disk, but checksum had not been updated.

(defmethod calculate-data-checksum ((di diskcopy-image))
  (let* ((ds (datasize di))
	 (ts (tagsize di))
	 (numwords (/ ds 2)))
    (calculate-checksum di 84 numwords)))



(defmethod calculate-tags-checksum ((di diskcopy-image))
  (let* ((ds (datasize di))
	 (ts (tagsize di))
	 (numwords (/ ts 2)))
  (calculate-checksum di (+ 84 ds) numwords)))

(defmethod block-count ((di diskcopy-image))
  (/ (datasize di) +dc42-block-length+))


;;;; diskcopy prodos support
;;;; see also dskimage.lisp
;;;;


(defmethod prodos-block ((di diskcopy-image) block-number)
  (unless (and (<= 0 block-number)
	       (< block-number (block-count di)))
    (error "ProDOS block number out of range."))
  (userdata-block di block-number))